/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
namespace Inet.Viewer.Data
{
    /// <summary>
    /// A SearchLoader instance loads the search result data from the server, parses it and forwards the results 
    /// to a specified <see cref="ISearchResultReceiver"/>.
    /// </summary>
    public class SearchLoader : Loader
    {
        private ISearchResultReceiver resultReceiver;
        private string phrase;
        private int startPage;
        private SearchOption flags;
        private bool isCanceled;
        private IRenderData reportData;

        /// <summary>
        /// Creates a search loader for a specified search.
        /// </summary>
        /// <param name="resultReceiver">the receiver which will get the results of the search</param>
        /// <param name="reportData">the data of the report where the search takes place</param>
        /// <param name="phrase">the search string</param>
        /// <param name="startPage">the search page to request from the server. 0 for a new search.</param>
        /// <param name="flags">addition flags, like case sensitivity</param>
        public SearchLoader(ISearchResultReceiver resultReceiver, IRenderData reportData, string phrase, int startPage, SearchOption flags)
        {
            this.resultReceiver = resultReceiver;
            this.reportData = reportData;
            this.phrase = phrase;
            this.startPage = startPage;
            this.flags = flags;
        }

        /// <summary>
        /// Performs the search.
        /// </summary>
        public virtual void Run()
        {
            Data = reportData.Search(phrase, startPage, flags);
            ReadTokens();
        }

        /// <inheritdoc />
        override protected internal bool Load(int token)
        {
            if (isCanceled)
            {
                return false;
            }
            switch (token)
            {
                case ViewerTokenConstants.TokenSearchResult:

                    int page = ReadInt();
                    string preContext = ReadString();
                    string result = ReadString();
                    string postContext = ReadString();
                    int numberOfChunks = ReadInt();
                    SearchChunk[] chunks = new SearchChunk[numberOfChunks];
                    int i = 0;

                    while (Offset < OldOffset + TokenSize && i < numberOfChunks)
                    {
                        int startIndex = ReadInt();
                        int endIndex = ReadInt();
                        int x = ReadInt();
                        int y = ReadInt();

                        chunks[i] = new SearchChunk(startIndex, endIndex, x, y, page);
                        i++;
                    }
                    resultReceiver.AddSearchResult(preContext, result, postContext, chunks);
                    return true;
                case ViewerTokenConstants.TokenSearchStatus:
                    int lastPage = ReadInt();
                    long timestamp = Read8ByteLong();
                    resultReceiver.EndSearch(lastPage, timestamp);
                    return true;
            }
            return base.Load(token);
        }

        /// <inheritdoc /> 
        public void Cancel()
        {
            isCanceled = true;
        }
    }
}